<?php
/*
Plugin Name: Text Sitemap Generator
Description: WordPressのルートディレクトリまたはサブディレクトリにテキスト形式のサイトマップ（urllist.txt）を生成します。
Version: 1.6
Author: KANAERU WEB
Author URI: https://kanaeru-web.biz/
License: GPL2
*/

// プラグインが有効化された時の設定
register_activation_hook(__FILE__, 'sitemap_plugin_activation');
function sitemap_plugin_activation() {
    update_option('sitemap_output_option', 2); // 初期設定で「カテゴリーのみ」を選択
    update_option('sitemap_output_location', 'root'); // 初期設定でルートディレクトリに設定
}

// サブディレクトリインストールの確認
function is_subdirectory_install() {
    $wordpress_address = get_option('siteurl');
    return trim(parse_url($wordpress_address, PHP_URL_PATH), '/') !== '';
}

// サブディレクトリ名取得
function get_subdirectory_name() {
    return trim(parse_url(get_option('siteurl'), PHP_URL_PATH), '/');
}

// ルートディレクトリのパスを取得（キャッシュ対応）
function get_root_directory_path() {
    static $root_dir = null;
    if ($root_dir === null) {
        $root_dir = ABSPATH;
        $subdir = get_subdirectory_name();
        if ($subdir) {
            $root_dir = rtrim(str_replace("/$subdir", '', $root_dir), '/');
        }
    }
    return $root_dir;
}

// ルートURLの取得
function get_root_directory_url() {
    static $site_url = null;
    if ($site_url === null) {
        $site_url = get_option('siteurl');
        $subdir = get_subdirectory_name();
        if ($subdir) {
            $site_url = str_replace("/$subdir", '', $site_url);
        }
    }
    return $site_url;
}

// サイトマップの保存パス取得
function get_sitemap_path() {
    return (get_option('sitemap_output_location', 'root') === 'root')
        ? get_root_directory_path() . '/urllist.txt'
        : ABSPATH . '/urllist.txt';
}

// サイトマップ生成機能（エラーハンドリング付き）
function generate_site_url_txt() {
    $option = get_option('sitemap_output_option', 1);
    $max_urls = min(get_option('sitemap_max_urls', 5000), 50000); // 最大値制限
    $url_count = 0;
    $output_location = get_option('sitemap_output_location', 'root');
    $site_url = '';

    // ルートURLの追加
    if ($output_location === 'root') {
        $site_url .= get_root_directory_url() . "\n";
        $url_count++;
    }

    // サブディレクトリのURL追加
    if (is_subdirectory_install() && get_root_directory_url() !== home_url()) {
        $site_url .= home_url() . "\n";
        $url_count++;
    }

    $excluded_post_ids = array_filter(explode(',', get_option('sitemap_excluded_post_ids', '')));
    $excluded_category_ids = array_filter(explode(',', get_option('sitemap_excluded_category_ids', '')));
    $excluded_tag_ids = array_filter(explode(',', get_option('sitemap_excluded_tag_ids', '')));

    // フロントページIDを除外リストに追加
    $front_page_id = get_option('page_on_front');
    if ($front_page_id) $excluded_post_ids[] = $front_page_id;

    // カテゴリーの処理
    if ($option == 1 || $option == 2) {
        foreach (get_categories(['hide_empty' => false]) as $cat) {
            if ($url_count >= $max_urls) break;
            if (!in_array($cat->term_id, $excluded_category_ids) && get_posts(['category' => $cat->term_id, 'posts_per_page' => 1, 'post_status' => 'publish'])) {
                $site_url .= get_category_link($cat->term_id) . "\n";
                $url_count++;
            }
        }
    }

    // タグの処理
    if ($option == 1 || $option == 3) {
        foreach (get_tags(['hide_empty' => false]) as $tag) {
            if ($url_count >= $max_urls) break;
            if (!in_array($tag->term_id, $excluded_tag_ids) && get_posts(['tag_id' => $tag->term_id, 'posts_per_page' => 1, 'post_status' => 'publish'])) {
                $site_url .= get_tag_link($tag->term_id) . "\n";
                $url_count++;
            }
        }
    }

    // 記事と固定ページのURL追加
    $paged = 1;
    do {
        $args = [
            'post_type' => ['post', 'page'],
            'posts_per_page' => 100,
            'post_status' => 'publish',
            'paged' => $paged++,
            'exclude' => $excluded_post_ids
        ];
        $posts_array = get_posts($args);

        foreach ($posts_array as $post) {
            if ($url_count >= $max_urls) break 2;
            $site_url .= get_permalink($post) . "\n";
            $url_count++;
        }
    } while (!empty($posts_array) && $url_count < $max_urls);

    // 任意のURLを追加
    foreach (array_filter(array_map('trim', explode("\n", get_option('sitemap_additional_urls', '')))) as $additional_url) {
        if ($url_count >= $max_urls) break;
        $site_url .= esc_url($additional_url) . "\n";
        $url_count++;
    }

    // サイトマップファイルの書き込み処理
    $sitemap_path = get_sitemap_path();
    if ($file = fopen($sitemap_path, 'w')) {
        fwrite($file, $site_url);
        fclose($file);
    } else {
        wp_die('サイトマップファイルを書き込みできません。ファイル権限を確認してください。');
    }
}

// 投稿または固定ページが新規公開や公開状態に変更されたときのみサイトマップを更新
add_action('transition_post_status', 'conditionally_generate_site_url_txt', 10, 3);
function conditionally_generate_site_url_txt($new_status, $old_status, $post) {
    // 投稿タイプが「投稿」または「固定ページ」のみ対象
    if (!in_array($post->post_type, ['post', 'page'])) {
        return;
    }

    // 公開状態に変わった場合、または新規公開された場合にのみサイトマップを更新
    if (($old_status !== 'publish' && $new_status === 'publish') || ($old_status === 'draft' && $new_status === 'publish')) {
        generate_site_url_txt();
    }
}

// 記事が削除またはゴミ箱に移動されたときにサイトマップを更新
add_action('wp_trash_post', 'generate_site_url_txt');
add_action('before_delete_post', 'generate_site_url_txt');

// カテゴリーやタグの変更があったときにもサイトマップを更新
add_action('create_category', 'generate_site_url_txt');
add_action('delete_category', 'generate_site_url_txt');
add_action('edit_category', 'generate_site_url_txt');
add_action('create_post_tag', 'generate_site_url_txt');
add_action('delete_post_tag', 'generate_site_url_txt');
add_action('edit_post_tag', 'generate_site_url_txt');

// 管理ページの設定
add_action('admin_menu', 'sitemap_plugin_menu');
function sitemap_plugin_menu() {
    add_options_page('Text Sitemap Generator Settings', 'Text Sitemap Generator', 'manage_options', 'text-sitemap-generator', 'sitemap_plugin_settings_page');
}

function sitemap_plugin_settings_page() {
    ?>
    <div class="wrap">
    <h2>Text Sitemap Generator Settings</h2>
    <form method="post" action="<?php echo admin_url('admin-post.php'); ?>">
        <?php wp_nonce_field('update_sitemap_settings', 'sitemap_settings_nonce'); ?>
        <input type="hidden" name="action" value="save_sitemap_settings">
        <table class="form-table">
            <tr valign="top">
            <th scope="row">Text Sitemap 出力設定</th>
            <td>
                <select name="sitemap_output_option">
                    <option value="1" <?php selected(get_option('sitemap_output_option'), 1); ?>>カテゴリーとタグ</option>
                    <option value="2" <?php selected(get_option('sitemap_output_option'), 2); ?>>カテゴリーのみ</option>
                    <option value="3" <?php selected(get_option('sitemap_output_option'), 3); ?>>タグのみ</option>
                </select>
                <p class="description">トップページと記事の他に、サイトマップに含める内容を選択します。<br />※デフォルト：カテゴリーのみ</p>
             </td>
            </tr>

            <tr valign="top">
            <th scope="row">最大URL数</th>
            <td>
                <input type="number" name="sitemap_max_urls" value="<?php echo esc_attr(get_option('sitemap_max_urls', 50000)); ?>" max="50000" />
                <p class="description">サイトマップに含めるURLの最大数（MAX：50000）を設定します。<br />※ご注意：記事、カテゴリー、タグが大量に存在する場合、サーバーの高負荷の原因となります。</p>
            </td>
            </tr>

            <tr valign="top">
            <th scope="row">除外する投稿ID</th>
            <td>
                <input type="text" name="sitemap_excluded_post_ids" value="<?php echo esc_attr(get_option('sitemap_excluded_post_ids')); ?>" />
                <p class="description">サイトマップから除外する投稿のIDをカンマ区切りで入力します。<br />※例）1,2,3</p>
            </td>
            </tr>

            <tr valign="top">
            <th scope="row">除外するカテゴリーID</th>
            <td>
                <input type="text" name="sitemap_excluded_category_ids" value="<?php echo esc_attr(get_option('sitemap_excluded_category_ids')); ?>" />
                <p class="description">サイトマップから除外するカテゴリーのIDをカンマ区切りで入力します。<br />※例）1,2,3</p>
            </td>
            </tr>

            <tr valign="top">
            <th scope="row">除外するタグID</th>
            <td>
                <input type="text" name="sitemap_excluded_tag_ids" value="<?php echo esc_attr(get_option('sitemap_excluded_tag_ids')); ?>" />
                <p class="description">サイトマップから除外するタグのIDをカンマ区切りで入力します。<br />※例）1,2,3</p>
            </td>
            </tr>

            <?php if (is_subdirectory_install()): ?>
            <tr valign="top">
            <th scope="row">出力先の選択</th>
            <td>
                <select name="sitemap_output_location">
                    <option value="subdir" <?php selected(get_option('sitemap_output_location'), 'subdir'); ?>>サブディレクトリ</option>
                    <option value="root" <?php selected(get_option('sitemap_output_location'), 'root'); ?>>ルートディレクトリ</option>
                </select>
                <p class="description">サイトマップの出力先を選択します。<br>※サブディレクトリにWordPressをインストールした環境のみのオプションです。</p>
            </td>
            </tr>
            <?php endif; ?>

            <tr valign="top">
            <th scope="row">任意のURLを追加</th>
            <td>
                <textarea name="sitemap_additional_urls" rows="5" cols="50"><?php echo esc_textarea(get_option('sitemap_additional_urls', '')); ?></textarea>
                <p class="description">1行につき1つのURLを入力してください。これらのURLはサイトマップに追加されます。</p>
            </td>
            </tr>

            <tr valign="top">
            <th scope="row">仮想robots.txtにサイトマップを追加</th>
            <td>
                <input type="checkbox" name="sitemap_add_to_robots" value="1" <?php checked(get_option('sitemap_add_to_robots'), 1); ?> />
                <label for="sitemap_add_to_robots">有効にする</label>
                <p class="description">WordPressが自動で生成する仮想robots.txtにurllist.txtのURLを追加します。<br />※手動でrobots.txtを設定している場合は適用されません。</p>
            </td>
            </tr>
        </table>
        <?php submit_button(); ?>
    </form>
    <?php
    $sitemap_url = (get_option('sitemap_output_location', 'root') === 'root') ? get_root_directory_url() . '/urllist.txt' : home_url('/urllist.txt');
    echo '<p><a href="' . esc_url($sitemap_url) . '" target="_blank">urllist.txtを表示</a></p>';
    ?>
    </div>
    <?php
}

add_action('admin_init', 'sitemap_generator_admin_init');
function sitemap_generator_admin_init() {
    $fields = [
        'sitemap_output_option' => 'sanitize_text_field',
        'sitemap_excluded_post_ids' => 'sanitize_text_field',
        'sitemap_excluded_category_ids' => 'sanitize_text_field',
        'sitemap_excluded_tag_ids' => 'sanitize_text_field',
        'sitemap_max_urls' => 'intval',
        'sitemap_output_location' => 'sanitize_text_field',
        'sitemap_additional_urls' => 'sanitize_textarea_field'
    ];
    foreach ($fields as $field => $sanitize_callback) {
        register_setting('text-sitemap-generator-options', $field, $sanitize_callback);
    }
}

// 設定保存時の処理
add_action('admin_post_save_sitemap_settings', 'handle_sitemap_settings_save');
function handle_sitemap_settings_save() {
    if (!current_user_can('manage_options')) {
        wp_die('権限がありません。');
    }
    check_admin_referer('update_sitemap_settings', 'sitemap_settings_nonce');

    // 古い出力先のパスを取得
    $old_output_location = get_option('sitemap_output_location', 'root');
    $old_sitemap_path = ($old_output_location === 'root') 
        ? get_root_directory_path() . '/urllist.txt' 
        : ABSPATH . '/urllist.txt';

    // 保存処理
    if (isset($_POST['sitemap_output_option'])) {
        update_option('sitemap_output_option', sanitize_text_field($_POST['sitemap_output_option']));
    }
    if (isset($_POST['sitemap_max_urls'])) {
        update_option('sitemap_max_urls', intval($_POST['sitemap_max_urls']));
    }
    if (isset($_POST['sitemap_excluded_post_ids'])) {
        update_option('sitemap_excluded_post_ids', sanitize_text_field($_POST['sitemap_excluded_post_ids']));
    }
    if (isset($_POST['sitemap_excluded_category_ids'])) {
        update_option('sitemap_excluded_category_ids', sanitize_text_field($_POST['sitemap_excluded_category_ids']));
    }
    if (isset($_POST['sitemap_excluded_tag_ids'])) {
        update_option('sitemap_excluded_tag_ids', sanitize_text_field($_POST['sitemap_excluded_tag_ids']));
    }
    if (isset($_POST['sitemap_output_location'])) {
        update_option('sitemap_output_location', sanitize_text_field($_POST['sitemap_output_location']));
    }
    if (isset($_POST['sitemap_additional_urls'])) {
        update_option('sitemap_additional_urls', sanitize_textarea_field($_POST['sitemap_additional_urls']));
    }

    // 出力先が変更された場合、古いファイルを削除
    $new_output_location = get_option('sitemap_output_location', 'root');
    if ($old_output_location !== $new_output_location && file_exists($old_sitemap_path)) {
        unlink($old_sitemap_path); // 古いファイルを削除
    }

    // サイトマップを再生成
    generate_site_url_txt();

    // 設定画面にリダイレクト
    wp_redirect(admin_url('options-general.php?page=text-sitemap-generator'));
    exit;
}

// robots.txtにサイトマップURLを追加
function add_sitemap_to_robots_txt($output, $public) {
    if (get_option('sitemap_add_to_robots') == 1) {
        $sitemap_url = (get_option('sitemap_output_location', 'root') === 'root') ? get_root_directory_url() . '/urllist.txt' : home_url('/urllist.txt');
        $output .= "Sitemap: " . esc_url($sitemap_url) . "\n";
    }
    return $output;
}
add_filter('robots_txt', 'add_sitemap_to_robots_txt', 9999, 2);

?>
