<?php
/*
Plugin Name: Auto Video Play & Stop
Description: 投稿・固定ページに挿入されたvideoタグを使用した動画に対して、スクロール位置に応じた自動再生・自動停止を実現します。再生・停止判定は「動画の高さ」または「ビューポートの高さ」で設定可能。オプションでcontrols属性付き動画も自動再生対象にできます。※jQuery（3.6.0以上推奨）が有効なテーマ・環境でご利用ください。
Version: 1.5.1
Author: KANAERU WEB
Author URI: https://kanaeru-web.biz/
License: GPL2
Requires at least: 5.5
Requires PHP: 7.4
*/

// 投稿本文にvideoタグがあるか検出（Gutenberg, Classic, Elementor HTMLウィジェット対応）
add_filter('the_content', function($content) {
	global $avps_has_video;
	if (strpos($content, '<video') !== false) {
		$avps_has_video = true;
	}
	return $content;
}, 1);

function avps_enqueue_scripts() {
	if (!is_admin()) {
		wp_enqueue_script('jquery');
		add_action('wp_footer', 'avps_check_and_insert_script', 9999);
	}
}
add_action('wp_enqueue_scripts', 'avps_enqueue_scripts');

function avps_check_and_insert_script() {
	if (!is_singular()) return;
	global $avps_has_video;
	$settings = get_option('avps_settings');
	$force_output = !empty($settings['avps_force_script_output']) && $settings['avps_force_script_output'] === 'yes';
	if ($force_output || !empty($avps_has_video)) avps_insert_script();
}

function avps_insert_script() {
	$settings = get_option('avps_settings');
	$vh = isset($settings['avps_visible_height']) ? intval($settings['avps_visible_height']) : 50;
	$vo = isset($settings['avps_visible_offset']) ? intval($settings['avps_visible_offset']) : 50;
	$vp_on = isset($settings['avps_editable_viewport_threshold']) && $settings['avps_editable_viewport_threshold'] === 'on';
	$vp_play = isset($settings['avps_viewport_play_threshold']) ? intval($settings['avps_viewport_play_threshold']) : 50;
	$vp_stop = isset($settings['avps_viewport_stop_threshold']) ? intval($settings['avps_viewport_stop_threshold']) : 50;
	$include_controls = isset($settings['avps_include_controls']) && $settings['avps_include_controls'] === 'yes';
?>
<script>
jQuery(function($) {
	var $window = $(window);
	var isInitialized = false;
	var useViewport = <?php echo $vp_on ? 'true' : 'false'; ?>;
	var playThreshold = <?php echo $vp_play; ?>;
	var stopThreshold = <?php echo $vp_stop; ?>;
	var visibleHeightRatio = <?php echo $vh / 100; ?>;
	var visibleOffsetRatio = <?php echo $vo / 100; ?>;
	var includeControls = <?php echo $include_controls ? 'true' : 'false'; ?>;

	function getAutoplayVideos() {
		return includeControls ? $('video') : $('video:not([controls])');
	}

	function handleVideoPlayback() {
		var $autoplayVideos = getAutoplayVideos();
		var scrollTop = $window.scrollTop();
		var winHeight = $window.innerHeight();
		$autoplayVideos.each(function() {
			var $v = $(this);
			var top = $v.offset().top;
			var height = $v.innerHeight();
			var bottom = top + height;
			var topIn, bottomIn;
			if (useViewport) {
				topIn = (top + winHeight * (playThreshold / 100)) < (scrollTop + winHeight);
				bottomIn = (bottom - winHeight * (stopThreshold / 100)) > scrollTop;
			} else {
				topIn = (top + height * visibleHeightRatio) < (scrollTop + winHeight);
				bottomIn = (bottom - height * visibleOffsetRatio) > scrollTop;
			}
			if (!$v.prop('paused') && (!topIn || !bottomIn)) {
				$v[0].pause();
			} else if ($v.prop('paused') && topIn && bottomIn) {
				$v[0].play();
			}
		});
	}

	function initializeVideoPlayback() {
		if (!isInitialized) {
			handleVideoPlayback();
			isInitialized = true;
		}
	}

	$window.on("scroll resize", handleVideoPlayback);
	initializeVideoPlayback();
});
</script>
<?php
}
// カスタム設定ページの追加
function avps_add_settings_page() {
	add_options_page(
		'Auto Video Play & Stop 設定',
		'Auto Video Play & Stop',
		'manage_options',
		'avps-settings',
		'avps_settings_page_callback'
	);
}
add_action('admin_menu', 'avps_add_settings_page');

function avps_settings_page_callback() {
	if (!current_user_can('manage_options')) {
		return;
	}
?>
<div class="wrap">
	<h1><?php echo esc_html(get_admin_page_title()); ?></h1>
	<form action="options.php" method="post">
		<?php
		settings_fields('avps_settings');
		do_settings_sections('avps_settings');
		submit_button('保存');
		?>
	</form>
</div>
<script>
  // 「手動で設定する」が変更されたときの処理（ビューポートの値をreadonly切り替え）
  document.getElementById("editableViewportThreshold").addEventListener("change", function () {
    var viewportPlayThreshold = document.getElementById("viewportPlayThreshold");
    var viewportStopThreshold = document.getElementById("viewportStopThreshold");
    if (this.checked) {
      viewportPlayThreshold.removeAttribute("readonly");
      viewportStopThreshold.removeAttribute("readonly");
    } else {
      viewportPlayThreshold.setAttribute("readonly", true);
      viewportStopThreshold.setAttribute("readonly", true);
    }

    // 動画高さのフィールドも切り替える
    toggleVideoHeightFields();
  });

  // 動画高さのしきい値からビューポートのしきい値を計算
  function updateViewportThresholds() {
    var visibleHeight = document.getElementById("visibleHeight").value;
    var visibleOffset = document.getElementById("visibleOffset").value;
    var viewportPlayThreshold = document.getElementById("viewportPlayThreshold");
    var viewportStopThreshold = document.getElementById("viewportStopThreshold");

    var calculatedPlayThreshold = (100 - visibleHeight) / 2;
    var calculatedStopThreshold = (100 - visibleOffset) / 2;

    viewportPlayThreshold.value = calculatedPlayThreshold;
    viewportStopThreshold.value = calculatedStopThreshold;
  }

  // 動画高さの入力欄の readonly 切り替え
  function toggleVideoHeightFields() {
    var editableCheckbox = document.getElementById("editableViewportThreshold");
    var visibleHeight = document.getElementById("visibleHeight");
    var visibleOffset = document.getElementById("visibleOffset");

    if (editableCheckbox.checked) {
      visibleHeight.setAttribute("readonly", true);
      visibleOffset.setAttribute("readonly", true);
    } else {
      visibleHeight.removeAttribute("readonly");
      visibleOffset.removeAttribute("readonly");
    }
  }

  // 初期計算
  updateViewportThresholds();

  // 初期のreadonly状態の設定
  toggleVideoHeightFields();

  // 動画高さのしきい値が変更されたときに再計算
  document.getElementById("visibleHeight").addEventListener("input", updateViewportThresholds);
  document.getElementById("visibleOffset").addEventListener("input", updateViewportThresholds);
</script>

<?php
}
// 設定の登録
function avps_register_settings() {
	register_setting(
		'avps_settings',
		'avps_settings',
		'avps_settings_sanitize_callback'
	);

	add_settings_section(
		'avps_settings_section',
		'動画の再生・停止のしきい値設定',
		'avps_settings_section_callback',
		'avps_settings'
	);

	add_settings_field(
		'avps_visible_height',
		'再生するしきい値(動画の高さ(％))',
		'avps_visible_height_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_visible_offset',
		'停止するしきい値(動画の高さ(％))',
		'avps_visible_offset_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_force_script_output',
		'スクリプト出力を常に有効にする',
		'avps_force_script_output_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_editable_viewport_threshold',
		'再生・停止のしきい値をビューポートで指定する',
		'avps_editable_viewport_threshold_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_viewport_play_threshold',
		'再生するしきい値(ビューポートの高さ(％))',
		'avps_viewport_play_threshold_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_viewport_stop_threshold',
		'停止するしきい値(ビューポートの高さ(％))',
		'avps_viewport_stop_threshold_callback',
		'avps_settings',
		'avps_settings_section'
	);

	add_settings_field(
		'avps_include_controls',
		'controls付き動画も自動再生対象にする',
		'avps_include_controls_callback',
		'avps_settings',
		'avps_settings_section'
	);
}
add_action('admin_init', 'avps_register_settings');

// 設定ページのセクション冒頭に表示する説明文
function avps_settings_section_callback() {
	echo '<p><strong>videoタグを使用した動画の自動再生・自動停止の条件を設定できます。</strong></p>';
	echo '<ul style="list-style-type: disc; padding-left: 20px;">';
	echo '<li>スクロール操作に応じて、動画が画面内に入ると再生、画面外に出ると自動停止します。</li>';
	echo '<li>しきい値の設定を自動計算する仕様ですが、「手動で設定する」にチェックを入れることでビューポートの高さで調整も可能です。</li>';
	echo '<li>通常、<code>controls</code> 属性が付いた動画は対象外ですが、下記オプションで自動再生に含めることができます。</li>';
	echo '<li>Elementorなどのページビルダーをお使いの場合、動画が検出されないケースがあります。その際は「スクリプト出力を常に有効にする」にチェックを入れてください。</li>';
	echo '<li><strong>jQuery（3.6.0以上推奨）</strong>が有効なテーマ・環境でご利用ください。</li>';
	echo '</ul>';
}

// 動画の再生開始を判定する「表示領域の高さ（％）」の入力フィールドを出力する
function avps_visible_height_callback() {
	$settings = get_option('avps_settings');
	$value = isset($settings['avps_visible_height']) ? $settings['avps_visible_height'] : 50;
	echo '<input type="number" id="visibleHeight" min="0" max="100" step="1" name="avps_settings[avps_visible_height]" value="' . esc_attr($value) . '"> %';
}

// 動画の停止を判定する「表示領域のオフセット（％）」の入力フィールドを出力する
function avps_visible_offset_callback() {
	$settings = get_option('avps_settings');
	$value = isset($settings['avps_visible_offset']) ? $settings['avps_visible_offset'] : 50;
	echo '<input type="number" id="visibleOffset" min="0" max="100" step="1" name="avps_settings[avps_visible_offset]" value="' . esc_attr($value) . '"> %';
}

// 「videoタグの有無に関係なくスクリプトを出力する」オプションのチェックボックスを出力する
function avps_force_script_output_callback() {
	$settings = get_option('avps_settings');
	$checked = isset($settings['avps_force_script_output']) && $settings['avps_force_script_output'] === 'yes';
	echo '<label><input type="checkbox" name="avps_settings[avps_force_script_output]" value="yes" ' . checked($checked, true, false) . '> videoタグの有無に関係なくスクリプトを出力する</label>';
	echo '<p><small>※チェックを入れた場合、すべてのページで動画再生・停止用のスクリプトが出力されます。</small></p>';
}

// 手動で設定するのコールバック関数
function avps_editable_viewport_threshold_callback() {
	$settings = get_option('avps_settings');
	$isEditable = isset($settings['avps_editable_viewport_threshold']) && $settings['avps_editable_viewport_threshold'] === 'on';
	echo '<input type="checkbox" id="editableViewportThreshold" name="avps_settings[avps_editable_viewport_threshold]" ' . checked($isEditable, true, false) . '> 手動で設定する';
	echo '<p><small><strong>※上級者向け：</strong>再生・停止位置を、「ビューポートの高さ(％)」で調整できます。なお、動画の高さによる制御は無効になります。</small></p>';
}

// ビューポートの高さの再生しきい値フィールドのコールバック関数
function avps_viewport_play_threshold_callback() {
	$settings = get_option('avps_settings');
	$value = isset($settings['avps_viewport_play_threshold']) ? $settings['avps_viewport_play_threshold'] : 50;
	$isEditable = isset($settings['avps_editable_viewport_threshold']) && $settings['avps_editable_viewport_threshold'] === 'on';
	echo '<input type="number" id="viewportPlayThreshold" min="0" max="100" step="1" name="avps_settings[avps_viewport_play_threshold]" value="' . esc_attr($value) . '" ' . ($isEditable ? '' : 'readonly') . '> %';
}

// ビューポートの高さの停止しきい値フィールドのコールバック関数
function avps_viewport_stop_threshold_callback() {
	$settings = get_option('avps_settings');
	$value = isset($settings['avps_viewport_stop_threshold']) ? $settings['avps_viewport_stop_threshold'] : 50;
	echo '<input type="number" id="viewportStopThreshold" min="0" max="100" step="1" name="avps_settings[avps_viewport_stop_threshold]" value="' . esc_attr($value) . '" readonly> %';
}

// 「controls属性付き動画も自動再生の対象に含める」オプションのチェックボックスを出力する
function avps_include_controls_callback() {
	$settings = get_option('avps_settings');
	$checked = isset($settings['avps_include_controls']) && $settings['avps_include_controls'] === 'yes';
	echo '<label><input type="checkbox" name="avps_settings[avps_include_controls]" value="yes" ' . checked($checked, true, false) . '> controls属性付き動画も自動再生の対象に含める</label>';
	echo '<p><small><strong>※上級者向け：</strong>一部のブラウザ（特にスマートフォン環境）では、controls属性が付いた動画の自動再生がブロックされる場合があります。</small></p>';
}

function avps_settings_sanitize_callback($input) {
	$new_input = [];

	if (isset($input['avps_visible_height'])) {
		$new_input['avps_visible_height'] = intval($input['avps_visible_height']);
	}

	if (isset($input['avps_visible_offset'])) {
		$new_input['avps_visible_offset'] = intval($input['avps_visible_offset']);
	}

	if (isset($input['avps_force_script_output'])) {
		$new_input['avps_force_script_output'] = sanitize_text_field($input['avps_force_script_output']);
	}

	if (isset($input['avps_editable_viewport_threshold'])) {
		$new_input['avps_editable_viewport_threshold'] = sanitize_text_field($input['avps_editable_viewport_threshold']);
	}

	if (isset($input['avps_viewport_play_threshold'])) {
		$new_input['avps_viewport_play_threshold'] = intval($input['avps_viewport_play_threshold']);
	}

	if (isset($input['avps_viewport_stop_threshold'])) {
		$new_input['avps_viewport_stop_threshold'] = intval($input['avps_viewport_stop_threshold']);
	}

	if (isset($input['avps_include_controls'])) {
		$new_input['avps_include_controls'] = sanitize_text_field($input['avps_include_controls']);
	}

	return $new_input;
}

// プラグイン一覧に「設定」のリンク追加
function avps_add_settings_link($links) {
	$settings_link = '<a href="options-general.php?page=avps-settings">' . __('設定') . '</a>';
	array_unshift($links, $settings_link);
	return $links;
}

$plugin = plugin_basename(__FILE__);
add_filter("plugin_action_links_$plugin", 'avps_add_settings_link');
